$scriptVer = '3.1'

configuration StigDsc
{
    [CmdletBinding()]
    param (
        [Parameter()]
        [String]
        $computerName = 'localhost'
    )

    Import-DscResource -ModuleName PSDscResources
    Import-DscResource -ModuleName SecurityPolicyDsc

    node $computerName
    {
        $domain = (Get-WmiObject win32_computersystem).Domain
        $64Bit = [Environment]::Is64BitOperatingSystem

        # Region Windows 10/11 STIG
        # Secondary Logon Service Disabled
        Service SecondaryLogon
        {
            Name        = 'seclogon'
            StartupType = 'Disabled'
            State       = 'Stopped'
        }

        # Turn on Basic feed authentication over HTTP
        Registry RssHttpAuth
        {
            Key       = 'HKLM:\Software\Policies\Microsoft\Internet Explorer\Feeds'
            ValueName = 'AllowBasicAuthInClear'
            Ensure    = 'Present'
            ValueData = '00000000'
            ValueType = 'DWORD'
            Force     = $true
        }

        # System objects: Strengthen default permissions of internal system objects (e.g. Symbolic Links)
        Registry DefaultPermissionsInternalSystemObjects
        {
            Key       = 'HKLM:\SYSTEM\CurrentControlSet\Control\Session Manager\'
            ValueName = 'ProtectionMode'
            Ensure    = 'Present'
            ValueData = '1'
            ValueType = 'DWORD'
            Force     = $true
        }

        if ($domain -ne 'WORKGROUP')
        {
            # Deny log on as a batch job
            UserRightsAssignment DenyLogonBatch
            {
                Policy   = 'Deny_log_on_as_a_batch_job'
                Identity = "$domain\Enterprise Admins","$domain\Domain Admins"
            }

            # Deny log on as a service
            UserRightsAssignment DenyLogonService
            {
                Policy   = 'Deny_log_on_as_a_service'
                Identity = "$domain\Enterprise Admins","$domain\Domain Admins"
            }
        }

        <#Account lockout duration
        Account lockout threshold
        Reset account lockout counter after
        Password must meet complexity requirements
        Store passwords using reversible encryption
        #>
        AccountPolicy AccountPolicies
        {
            Name                                        = 'STIGs'
            Password_must_meet_complexity_requirements  = 'Enabled'
            Store_passwords_using_reversible_encryption = 'Disabled'
            Account_lockout_duration                    = 15
            Account_lockout_threshold                   = 3
            Reset_account_lockout_counter_after         = 15
        }

        # Set security options not available in Intune
        SecurityOption AccountSecurityOptions
        {
            Name                                                             = 'STIG Secutity Options'
            Network_access_Allow_anonymous_SID_Name_translation              = 'Disabled'
            Network_access_Let_Everyone_permissions_apply_to_anonymous_users = 'Disabled'
        }

        # Remove "Run As Different User" from context menus
        Registry NoStartBanner
        {
            Key       = 'HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Policies\Explorer'
            ValueName = 'NoStartBanner'
            Ensure    = 'Present'
            ValueData = '00000001'
            ValueType = 'DWORD'
            Force     = $true
        }

        Registry BatSuppression
        {
            Key       = 'HKLM:\SOFTWARE\Classes\batfile\shell\runasuser'
            ValueName = 'SuppressionPolicy'
            Ensure    = 'Present'
            ValueData = '4096'
            ValueType = 'DWORD'
            Force     = $true
        }

        Registry CmdSuppression
        {
            Key       = 'HKLM:\SOFTWARE\Classes\cmdfile\shell\runasuser'
            ValueName = 'SuppressionPolicy'
            Ensure    = 'Present'
            ValueData = '4096'
            ValueType = 'DWORD'
            Force     = $true
        }

        Registry ExeSuppression
        {
            Key       = 'HKLM:\SOFTWARE\Classes\exefile\shell\runasuser'
            ValueName = 'SuppressionPolicy'
            Ensure    = 'Present'
            ValueData = '4096'
            ValueType = 'DWORD'
            Force     = $true
        }

        Registry MscSuppression
        {
            Key       = 'HKLM:\SOFTWARE\Classes\mscfile\shell\runasuser'
            ValueName = 'SuppressionPolicy'
            Ensure    = 'Present'
            ValueData = '4096'
            ValueType = 'DWORD'
            Force     = $true
        }

        # Network security: LDAP client signing requirements
        Registry LDAPClientSigningRequirements
        {
            Key       = 'HKLM:\SYSTEM\CurrentControlSet\Services\LDAP\'
            ValueName = 'LDAPClientIntegrity'
            Ensure    = 'Present'
            ValueData = '1'
            ValueType = 'DWORD'
            Force     = $true
        }

        # Network security: Configure encryption types allowed for Kerberos
        Registry EncryptionTypesKerberos
        {
            Key       = 'HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Policies\System\Kerberos\Parameters\'
            ValueName = 'SupportedEncryptionTypes'
            Ensure    = 'Present'
            ValueData = '2147483640'
            ValueType = 'DWORD'
            Force     = $true
        }

        # Network security: Allow LocalSystem NULL session fallback
        Registry NULLSessionFallback
        {
            Key       = 'HKLM:\SYSTEM\CurrentControlSet\Control\LSA\MSV1_0\'
            ValueName = 'allownullsessionfallback'
            Ensure    = 'Present'
            ValueData = '0'
            ValueType = 'DWORD'
            Force     = $true
        }

        # Interactive logon: Number of previous logons to cache (in case domain controller is not available)
        Registry PreviousLogonsToCache
        {
            Key       = 'HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion\Winlogon\'
            ValueName = 'CachedLogonsCount'
            Ensure    = 'Present'
            ValueData = '10'
            ValueType = 'String'
            Force     = $true
        }

        # Domain member: Require strong (Windows 2000 or later) session key
        Registry RequireStrongSessionKey
        {
            Key       = 'HKLM:\SYSTEM\CurrentControlSet\Services\Netlogon\Parameters\'
            ValueName = 'RequireStrongKey'
            Ensure    = 'Present'
            ValueData = '1'
            ValueType = 'DWORD'
            Force     = $true
        }

        # Domain member: Disable machine account password changes
        Registry DisableMachineAccountPassword
        {
            Key       = 'HKLM:\SYSTEM\CurrentControlSet\Services\Netlogon\Parameters\'
            ValueName = 'DisablePasswordChange'
            Ensure    = 'Present'
            ValueData = '0'
            ValueType = 'DWORD'
            Force     = $true
        }

        # Domain member: Maximum machine account password age
        Registry MaxMachineAccountPasswordAge
        {
            Key       = 'HKLM:\SYSTEM\CurrentControlSet\Services\Netlogon\Parameters\'
            ValueName = 'MaximumPasswordAge'
            Ensure    = 'Present'
            ValueData = '30'
            ValueType = 'DWORD'
            Force     = $true
        }

        # Domain member: Digitally encrypt secure channel data (when possible)
        Registry DigitallyEncryptSecureChannel
        {
            Key       = 'HKLM:\SYSTEM\CurrentControlSet\Services\Netlogon\Parameters\'
            ValueName = 'SealSecureChannel'
            Ensure    = 'Present'
            ValueData = '1'
            ValueType = 'DWORD'
            Force     = $true
        }

        # Domain member: Digitally sign secure channel data (when possible)
        Registry DigitallySignSecureChannel
        {
            Key       = 'HKLM:\SYSTEM\CurrentControlSet\Services\Netlogon\Parameters\'
            ValueName = 'SignSecureChannel'
            Ensure    = 'Present'
            ValueData = '1'
            ValueType = 'DWORD'
            Force     = $true
        }

        # Audit: Force audit policy subcategory settings (Windows Vista or later) to override audit policy category settings
        Registry AuditPolicySubCategory
        {
            Key       = 'HKLM:\SYSTEM\CurrentControlSet\Control\Lsa\'
            ValueName = 'SCENoApplyLegacyAuditPolicy'
            Ensure    = 'Present'
            ValueData = '1'
            ValueType = 'DWORD'
            Force     = $true
        }
        # End Region Windows 10/11 STIG

        # Region IE 11 STIG
        # Disable Internet Explorer 11 as a standalone browser
        Registry IE11StandaloneBrowser
        {
            Key       = 'HKLM:\Software\Policies\Microsoft\Internet Explorer\Main'
            ValueName = 'NotifyDisableIEOptions'
            Ensure    = 'Present'
            ValueData = '2'
            ValueType = 'DWORD'
            Force     = $true
        }

        # Turn off Developer Tools
        Registry IE11DeveloperTools
        {
            Key       = 'HKLM:\Software\Policies\Microsoft\Internet Explorer\IEDevTools\'
            ValueName = 'Disabled'
            Ensure    = 'Present'
            ValueData = '1'
            ValueType = 'DWORD'
            Force     = $true
        }
        # End Region IE 11 STIG

        # Region Windows Firewall STIG
        # Windows Defender Firewall: Allow logging/Size limit (KB) Domain Profile
        Registry DomainLogFileSize
        {
            Key       = 'HKLM:\SOFTWARE\Policies\Microsoft\WindowsFirewall\DomainProfile\Logging\'
            ValueName = 'LogFileSize'
            Ensure    = 'Present'
            ValueData = '16384'
            ValueType = 'DWORD'
            Force     = $true
        }

        Registry DomainLogFileSize2
        {
            Key       = 'HKLM:\SYSTEM\CurrentControlSet\Services\SharedAccess\Parameters\FirewallPolicy\DomainProfile\Logging\'
            ValueName = 'LogFileSize'
            Ensure    = 'Present'
            ValueData = '16384'
            ValueType = 'DWORD'
            Force     = $true
        }

        # Windows Defender Firewall: Allow logging/Log dropped packets Domain Profile
        Registry DomainLogDroppedPackets
        {
            Key       = 'HKLM:\SOFTWARE\Policies\Microsoft\WindowsFirewall\DomainProfile\Logging\'
            ValueName = 'LogDroppedPackets'
            Ensure    = 'Present'
            ValueData = '1'
            ValueType = 'DWORD'
            Force     = $true
        }

        Registry DomainLogDroppedPackets2
        {
            Key       = 'HKLM:\SYSTEM\CurrentControlSet\Services\SharedAccess\Parameters\FirewallPolicy\DomainProfile\Logging\'
            ValueName = 'LogDroppedPackets'
            Ensure    = 'Present'
            ValueData = '1'
            ValueType = 'DWORD'
            Force     = $true
        }

        # Windows Defender Firewall: Allow logging/Log successful connections Domain Profile
        Registry DomainLogSuccessfulConnections
        {
            Key       = 'HKLM:\SOFTWARE\Policies\Microsoft\WindowsFirewall\DomainProfile\Logging\'
            ValueName = 'LogSuccessfulConnections'
            Ensure    = 'Present'
            ValueData = '1'
            ValueType = 'DWORD'
            Force     = $true
        }

        Registry DomainLogSuccessfulConnections2
        {
            Key       = 'HKLM:\SYSTEM\CurrentControlSet\Services\SharedAccess\Parameters\FirewallPolicy\DomainProfile\Logging\'
            ValueName = 'LogSuccessfulConnections'
            Ensure    = 'Present'
            ValueData = '1'
            ValueType = 'DWORD'
            Force     = $true
        }

        # Windows Defender Firewall: Allow logging/Size limit (KB) Private Profile
        Registry PrivateLogFileSize
        {
            Key       = 'HKLM:\SOFTWARE\Policies\Microsoft\WindowsFirewall\PrivateProfile\Logging\'
            ValueName = 'LogFileSize'
            Ensure    = 'Present'
            ValueData = '16384'
            ValueType = 'DWORD'
            Force     = $true
        }

        Registry PrivateLogFileSize2
        {
            Key       = 'HKLM:\SYSTEM\CurrentControlSet\Services\SharedAccess\Parameters\FirewallPolicy\StandardProfile\Logging\'
            ValueName = 'LogFileSize'
            Ensure    = 'Present'
            ValueData = '16384'
            ValueType = 'DWORD'
            Force     = $true
        }

        # Windows Defender Firewall: Allow logging/Log dropped packets Private Profile
        Registry PrivateLogDroppedPackets
        {
            Key       = 'HKLM:\SOFTWARE\Policies\Microsoft\WindowsFirewall\PrivateProfile\Logging\'
            ValueName = 'LogDroppedPackets'
            Ensure    = 'Present'
            ValueData = '1'
            ValueType = 'DWORD'
            Force     = $true
        }

        Registry PrivateLogDroppedPackets2
        {
            Key       = 'HKLM:\SYSTEM\CurrentControlSet\Services\SharedAccess\Parameters\FirewallPolicy\StandardProfile\Logging\'
            ValueName = 'LogDroppedPackets'
            Ensure    = 'Present'
            ValueData = '1'
            ValueType = 'DWORD'
            Force     = $true
        }

        # Windows Defender Firewall: Allow logging/Log successful connections Private Profile
        Registry PrivateLogSuccessfulConnections
        {
            Key       = 'HKLM:\SOFTWARE\Policies\Microsoft\WindowsFirewall\PrivateProfile\Logging\'
            ValueName = 'LogSuccessfulConnections'
            Ensure    = 'Present'
            ValueData = '1'
            ValueType = 'DWORD'
            Force     = $true
        }

        Registry PrivateLogSuccessfulConnections2
        {
            Key       = 'HKLM:\SYSTEM\CurrentControlSet\Services\SharedAccess\Parameters\FirewallPolicy\StandardProfile\Logging\'
            ValueName = 'LogSuccessfulConnections'
            Ensure    = 'Present'
            ValueData = '1'
            ValueType = 'DWORD'
            Force     = $true
        }

        # Windows Defender Firewall: Allow logging/Size limit (KB) Public Profile
        Registry PublicLogFileSize
        {
            Key       = 'HKLM:\SOFTWARE\Policies\Microsoft\WindowsFirewall\PublicProfile\Logging\'
            ValueName = 'LogFileSize'
            Ensure    = 'Present'
            ValueData = '16384'
            ValueType = 'DWORD'
            Force     = $true
        }

        Registry PublicLogFileSize2
        {
            Key       = 'HKLM:\SYSTEM\CurrentControlSet\Services\SharedAccess\Parameters\FirewallPolicy\PublicProfile\Logging\'
            ValueName = 'LogFileSize'
            Ensure    = 'Present'
            ValueData = '16384'
            ValueType = 'DWORD'
            Force     = $true
        }

        # Windows Defender Firewall: Allow logging/Log dropped packets Public Profile
        Registry PublicLogDroppedPackets
        {
            Key       = 'HKLM:\SOFTWARE\Policies\Microsoft\WindowsFirewall\PublicProfile\Logging\'
            ValueName = 'LogDroppedPackets'
            Ensure    = 'Present'
            ValueData = '1'
            ValueType = 'DWORD'
            Force     = $true
        }

        Registry PublicLogDroppedPackets2
        {
            Key       = 'HKLM:\SYSTEM\CurrentControlSet\Services\SharedAccess\Parameters\FirewallPolicy\PublicProfile\Logging\'
            ValueName = 'LogDroppedPackets'
            Ensure    = 'Present'
            ValueData = '1'
            ValueType = 'DWORD'
            Force     = $true
        }

        # Windows Defender Firewall: Allow logging/Log successful connections Public Profile
        Registry PublicLogSuccessfulConnections
        {
            Key       = 'HKLM:\SOFTWARE\Policies\Microsoft\WindowsFirewall\PublicProfile\Logging\'
            ValueName = 'LogSuccessfulConnections'
            Ensure    = 'Present'
            ValueData = '1'
            ValueType = 'DWORD'
            Force     = $true
        }

        Registry PublicLogSuccessfulConnections2
        {
            Key       = 'HKLM:\SYSTEM\CurrentControlSet\Services\SharedAccess\Parameters\FirewallPolicy\PublicProfile\Logging\'
            ValueName = 'LogSuccessfulConnections'
            Ensure    = 'Present'
            ValueData = '1'
            ValueType = 'DWORD'
            Force     = $true
        }

        # WNDF-AV-000070 enable Endpoint detection and response and set to block mode
        Registry PassiveRemediation
        {
            Key       = 'HKLM:\Software\Policies\Microsoft\Windows Defender\Features'
            ValueName = 'PassiveRemediation'
            Ensure    = 'Present'
            ValueData = '1'
            ValueType = 'DWORD'
            Force     = $true
        }

        # End Region Windows Firewall STIG

        # Region Office 2019 M365 Apps
        # Block Flash activation in Office documents
        Registry OfficeFilterOverride
        {
            Key       = 'HKLM:\SOFTWARE\Microsoft\Office\Common\COM Compatibility\'
            ValueName = 'ActivationFilterOverride'
            Ensure    = 'Present'
            ValueData = '0'
            ValueType = 'DWORD'
            Force     = $true
        }

        Registry OfficeCompatFlags
        {
            Key       = 'HKLM:\SOFTWARE\Microsoft\Office\Common\COM Compatibility\'
            ValueName = 'Compatibility Flags'
            Ensure    = 'Present'
            ValueData = '1024'
            ValueType = 'DWORD'
            Force     = $true
        }
        # End Region Office 2019 M365 Apps

        # Region Google Chrome
        # Auto-update check period override
        Registry AutoUpdateCheckPeriodMinutes
        {
            Key       = 'HKLM:\Software\Policies\Google\Update\'
            ValueName = 'AutoUpdateCheckPeriodMinutes'
            Ensure    = 'Present'
            ValueData = '10080'
            ValueType = 'DWORD'
            Force     = $true
        }

        Registry CreateThemesSettings
        {
            Key       = 'HKLM:\Software\Policies\Google\Chrome'
            ValueName = 'CreateThemesSettings'
            Ensure    = 'Present'
            ValueData = '2'
            ValueType = 'DWORD'
            Force     = $true
        }

        Registry DevToolsGenAiSettings
        {
            Key       = 'HKLM:\Software\Policies\Google\Chrome'
            ValueName = 'DevToolsGenAiSettings'
            Ensure    = 'Present'
            ValueData = '2'
            ValueType = 'DWORD'
            Force     = $true
        }

        Registry GenAILocalFoundationalModelSettings
        {
            Key       = 'HKLM:\Software\Policies\Google\Chrome'
            ValueName = 'GenAILocalFoundationalModelSettings'
            Ensure    = 'Present'
            ValueData = '1'
            ValueType = 'DWORD'
            Force     = $true
        }

        Registry HelpMeWriteSettings
        {
            Key       = 'HKLM:\Software\Policies\Google\Chrome'
            ValueName = 'HelpMeWriteSettings'
            Ensure    = 'Present'
            ValueData = '2'
            ValueType = 'DWORD'
            Force     = $true
        }

        Registry HistorySearchSettings
        {
            Key       = 'HKLM:\Software\Policies\Google\Chrome'
            ValueName = 'HistorySearchSettings'
            Ensure    = 'Present'
            ValueData = '2'
            ValueType = 'DWORD'
            Force     = $true
        }

        Registry TabCompareSettings
        {
            Key       = 'HKLM:\Software\Policies\Google\Chrome'
            ValueName = 'TabCompareSettings'
            Ensure    = 'Present'
            ValueData = '2'
            ValueType = 'DWORD'
            Force     = $true
        }
        # End Region Google Chrome

        # Region .NET STIG
        # V-81495: Disable TLS RC4 cipher in .NET
        if ($64Bit -eq $true)
        {
            Registry SchUseStrongCrypto2
            {
                Key       = 'HKLM:\SOFTWARE\Microsoft\.NETFramework\v2.0.50727\'
                ValueName = 'SchUseStrongCrypto'
                Ensure    = 'Present'
                ValueData = '00000001'
                ValueType = 'DWORD'
                Force     = $true
            }

            Registry SchUseStrongCrypto4
            {
                Key       = 'HKLM:\SOFTWARE\Microsoft\.NETFramework\v4.0.30319\'
                ValueName = 'SchUseStrongCrypto'
                Ensure    = 'Present'
                ValueData = '00000001'
                ValueType = 'DWORD'
                Force     = $true
            }

            Registry SchUseStrongCrypto264
            {
                Key       = 'HKLM:\SOFTWARE\Wow6432Node\Microsoft\.NETFramework\v2.0.50727\'
                ValueName = 'SchUseStrongCrypto'
                Ensure    = 'Present'
                ValueData = '00000001'
                ValueType = 'DWORD'
                Force     = $true
            }

            Registry SchUseStrongCrypto464
            {
                Key       = 'HKLM:\SOFTWARE\Wow6432Node\Microsoft\.NETFramework\v4.0.30319\'
                ValueName = 'SchUseStrongCrypto'
                Ensure    = 'Present'
                ValueData = '00000001'
                ValueType = 'DWORD'
                Force     = $true
            }
        }
        else
        {
            Registry SchUseStrongCrypto2
            {
                Key       = 'HKLM:\SOFTWARE\Microsoft\.NETFramework\v2.0.50727\'
                ValueName = 'SchUseStrongCrypto'
                Ensure    = 'Present'
                ValueData = '00000001'
                ValueType = 'DWORD'
                Force     = $true
            }

            Registry SchUseStrongCrypto4
            {
                Key       = 'HKLM:\SOFTWARE\Microsoft\.NETFramework\v4.0.30319\'
                ValueName = 'SchUseStrongCrypto'
                Ensure    = 'Present'
                ValueData = '00000001'
                ValueType = 'DWORD'
                Force     = $true
            }
        }
        # End Region .NET STIG

        # Region Version Control
        Registry DscVersion
        {
            Key       = 'HKLM:\SOFTWARE\DisaSTIG'
            ValueName = 'DSCVersion'
            Ensure    = 'Present'
            ValueData = $scriptVer
            ValueType = 'String'
            Force     = $true
        }
        # End Region Version Control
    }
}

# LocalConfigurationManager Config
[DscLocalConfigurationManager()]
Configuration LCM
{
    Node localhost
    {
        Settings
        {
            ConfigurationMode         = 'ApplyAndAutoCorrect'
            StatusRetentionTimeInDays = 7
        }
    }
}

# Generate Mof and Meta Mof
StigDsc -OutputPath 'C:\Windows\SysWOW64\DISASTIGs'
LCM -OutputPath 'C:\Windows\SysWOW64\DISASTIGs'

# Start LCM Configuration and DSC Apply
Set-DscLocalConfigurationManager -Path 'C:\Windows\SysWOW64\DISASTIGs'
Start-DscConfiguration -Path 'C:\Windows\SysWOW64\DISASTIGs' -Wait -Verbose
